/* Copyright (c) 2001 EU DataGrid.
 * For license conditions see http://www.eu-datagrid.org/license.html
 */

import org.edg.info.Archiver;
import org.edg.info.DataBaseProducer;
import org.edg.info.LatestProducer;
import org.edg.info.RGMAException;
import org.edg.info.StreamProducer;
import java.util.Enumeration;
import java.util.Hashtable;

/**
 * History/Latest Archiver for a table (defined by the user).
 * Takes database details on command line.
 */
public class BossArchiver {    
    
    private Hashtable m_tables;
    private Archiver m_historyArchiver;
    private Archiver m_latestArchiver;
    private DataBaseProducer m_dataBaseProducer;
    private LatestProducer m_latestProducer;
    private String m_password;
    private String m_rdbms;
    private String m_user;
    private static BossArchiver m_bossArchiver;


    /**
     * XXX: NB!
     *
     * Add any table definitions here. When started, this program will
     * run this method to create the tables.
     */
    private void init() {

	m_tables = new Hashtable();

/*	// The table name to be archived  	
	String tableName = "bossJobExOutMessage";
	// The table description of the table, 
	//   this should match the description defined under bossJob  
	String bossJobExTable = "CREATE TABLE " + tableName + " " +
	    "(bossDatabaseHost VARCHAR(255) NOT NULL, " +
	    "bossDatabaseName VARCHAR(255) NOT NULL, " +
	    "bossJobId INT NOT NULL, " +
	    "bossJobtype VARCHAR(31), " +
	    "bossVarName VARCHAR(31) NOT NULL, " +
	    "bossVarValue VARCHAR(31), " +
	    "timestamp VARCHAR(15), " +
	    "MeasurementDate DATE NOT NULL, " +
	    "MeasurementTime TIME NOT NULL, " +
	    "PRIMARY KEY (bossDatabaseHost, bossDatabaseName, bossJobId, bossVarName))";
*/

	String tableName = "bossJobExOutMessage5";
	// The table description of the table, 
	//   this should match the description defined under bossJob 
	String bossJobExTable = "CREATE TABLE " + tableName + " " +
	    "(bossDatabaseHost VARCHAR(200) PRIMARY KEY, " +
	    "bossDatabaseName VARCHAR(200) PRIMARY KEY, " +
	    "bossJobId INT PRIMARY KEY, " +
	    "bossJobtype VARCHAR(31), " +
	    "bossVarName VARCHAR(31) PRIMARY KEY, " +
	    "bossVarValue VARCHAR(63), " +
	    "timestamp VARCHAR(15))";

	// Add the description into the hash table here
	m_tables.put(tableName, bossJobExTable);

    }

    /**
     * Default constructor creates an Archiver using only a LatestProducer.
     */
    public BossArchiver() throws RGMAException {

	init();
        m_latestProducer = new LatestProducer();
        m_latestArchiver = new Archiver(m_latestProducer);
        declareArchiverTables();

    }    


    /**
     * Creates a new HistoryArchiver object.
     *
     * @param p_rdbms Database URL
     * @param p_user DB User name
     * @param p_password DB password
     * @throws RGMAException Thrown if DBProducer or Archiver could not be instantiated.
     */
    public BossArchiver(String p_rdbms, String p_user,
        String p_password) throws RGMAException {
        
	init();
        createHistoryArchiver(p_rdbms, p_user, p_password);
        // Instantiate latest archiver
        m_latestProducer = new LatestProducer();
        m_latestArchiver = new Archiver(m_latestProducer);
        declareArchiverTables();
        
    }
    
    /**
     * Creates a new BossArchiver object.
     *
     * @param p_rdbms Database URL
     * @param p_user DB User name
     * @param p_password DB password
     * @throws RGMAException Thrown if DBProducer or Archiver could not be instantiated.
     */
    public BossArchiver(String p_hrdbms, String p_huser,
        String p_hpassword, String p_lprdbms, String p_lpuser,
        String p_lppassword) throws RGMAException {
        
 	init();       
	createHistoryArchiver(p_hrdbms, p_huser, p_hpassword);
        createLatestArchiver(p_lprdbms, p_lpuser, p_lppassword);
        declareArchiverTables();
        
    }

    private void createLatestArchiver(String p_lprdbms, String p_lpuser,
    	String p_lppassword) throws RGMAException {
        
	// Instantiate latest archiver
        m_latestProducer = new LatestProducer(p_lprdbms, p_lpuser, p_lppassword);
        m_latestArchiver = new Archiver(m_latestProducer);
 
    }

    private void declareArchiverTables() throws RGMAException {
    
        System.out.println("\nArchiving tables:");
 	Enumeration enum = m_tables.keys();       
	while (enum.hasMoreElements()) {
	    String tableName = enum.nextElement().toString();
	    String currentDesc = m_tables.get(tableName).toString();
	    System.out.print("\ttable description: " + currentDesc + ": ");
            System.out.println("table name: " + tableName);
	    if (m_historyArchiver != null) {
		/* Non-standard table needs a description: */
		m_historyArchiver.declareTable(tableName, "", currentDesc);
//		m_historyArchiver.declareTable(tableName, "");
            }
            StreamProducer t = new StreamProducer();
	    /* Non-standard table needs a description: */
	    t.declareTable(tableName, "", currentDesc);
//	    t.declareTable(tableName, "");
            t = null; 
	    /* Non-standard table needs a description: */
	    m_latestArchiver.declareTable(tableName, "", currentDesc);
//	    m_latestArchiver.declareTable(tableName, "");
            System.out.println("[LATEST] ");
        }

    }

    private void createHistoryArchiver(String p_rdbms, String p_user,
        String p_password)throws RGMAException {
                
        // Instantiate history archiver
        m_rdbms = p_rdbms;
        m_user = p_user;
        m_password = p_password;
        m_dataBaseProducer = new DataBaseProducer(m_rdbms, m_user, m_password);        
        m_historyArchiver = new Archiver(m_dataBaseProducer);
    
    }

    /**
     * Main program.
     *
     * @param args Command line arguments.
     */
    public static void main(String[] args) throws Exception {
        //LatestProducer lp = new LatestProducer();
	//lp.declareTable("ServiceStatus", "");
	System.out.println("BossArchiver");
 	System.out.println("number of args: " + args.length);       
        try {
            if (args.length == 6) {
                System.out.println("LatestProducer archiver:");
                System.out.println("\tDatabase URL: " + args[3]);
                System.out.println("\tUser: " + args[4]);
                System.out.println("\tPassword: " + args[5]);
                m_bossArchiver = new BossArchiver(args[0],
                    args[1], args[2], args[3], args[4], args[5]);
            } else if (args.length == 3) {
                m_bossArchiver = new BossArchiver(args[0],
                    args[1], args[2]);
            } else if (args.length == 0){
		m_bossArchiver = new BossArchiver();
	    } else {
                showSyntax();
                System.exit(2);
	    }
            Runtime.getRuntime().addShutdownHook(new Thread() {
                public void run() {
                   if (m_bossArchiver != null) {
                       m_bossArchiver.close();
                   }
                }
            });
            m_bossArchiver.run();
        } catch (RGMAException e) {
            e.printStackTrace();
        }
        if (m_bossArchiver != null) {
            m_bossArchiver.close();
        }

    }

    /**
     * Closes the databaseProducer and Archiver
     */
    public void close() {

        if (m_historyArchiver != null) {
            System.out.println("Shutting down history archiver.");
            m_historyArchiver.close();
        }
        if (m_latestArchiver != null) {
            System.out.println("Shutting down latest archiver.");
            m_latestArchiver.close();
        }

    }

    /**
     * Calls showSignOfLife every 10 minutes, otherwise sleeps.
     * Exits method if exception is thrown.
     */
    public void run() {

        try {
            while (true) {
                if (m_historyArchiver != null) {
		    m_historyArchiver.showSignOfLife();
                }
		m_latestArchiver.showSignOfLife();
                // Wait 3.3 minutes JJN
                Thread.sleep(200000);
            }
        } catch (RGMAException e) {
            System.err.println("Show sign of life failed, exiting.");
            e.printStackTrace();
        } catch (InterruptedException e) {
            System.err.println("Keep-alive thread interrupted, exiting.");
            e.printStackTrace();
        }

    }

    private static void showSyntax() {

        System.out.println("Usage:");
        System.out.println(
            "\tjava ServiceDataHistoryArchiver [default] | " +
	    "<h_dbURL> <h_user> <h_pass> [ <lp_dbURL> <lp_user> <lp_pass> ]");
        System.out.println("Parameters:");
        System.out.println("\tdefault: Creates one Archiver using a LatestProducer, " +
	    "database is dynamically created");
        System.out.println("\th_dbUrl: History database URL.");
        System.out.println("\th_user:  History DB username.");
        System.out.println("\th_pass:  History DB password.");
        System.out.println("\tlp_dbUrl: LatestProducer database URL.");
        System.out.println("\tlp_user:  LatestProducer DB username.");
        System.out.println("\tlp_pass:  LatestProducer DB password.");

    }
}
